#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

struct TPointerEntry
{
   int  loc;
   int  pointer;
   int  spriteNum;
   int  movementNum;
   int  size;
   char plainString[4096];
   char codeString[4096];
};

void          LoadCompressedText(FILE*);
void          LoadTextPointers(FILE*, TPointerEntry[], int&);
int           GetType(unsigned char ch);
void          GetStringAt(FILE*, unsigned int, char[], int, int&);
unsigned int  HexToSNES(unsigned int);
unsigned int  SNESToHex(unsigned int);
void          WriteText(FILE*, unsigned int, char*, int&);
int           ExpandROM(FILE*, unsigned int);
int           GetNumbers(char*, int[256]);
void          SimplifyString(char*, char[1024], int&);
int           CharToHex(char);

char          comprText[768][25];

int main(int argc, char* argv[])
{
   FILE*         rom;
   TPointerEntry tpe[3];
   int           tpeCount;
   unsigned int  tpAddr;

   rom = fopen("ebv2.smc", "rb+");
   LoadCompressedText(rom);

   for (int i = 0; i < 3; i++)
   {
      for (int j = 0; j < 255; j++)
      {
          printf("%.2X %.2X = %s\n", 21 + i, j, comprText[i * 256 + j]);
      }
   }

   /*
   LoadTextPointers(rom, tpe, tpeCount);

   for (int i = 0; i < tpeCount; i++)
   {
      tpAddr = SNESToHex(tpe[i].pointer);

      printf("0x%.6x -> 0x%.6x  0x%.6x  %.3d  %.3d  %s\n", tpe[i].loc, tpAddr, tpe[i].pointer, tpe[i].spriteNum, tpe[i].movementNum, tpe[i].plainString);
   }*/

   fclose(rom);
}

void LoadCompressedText(FILE* rom)
{
   // Loads the look-up table into the comprText list of strings.
   // Assumes rom is in read binary mode.

   unsigned char ch;
   char          strLoc;

   comprText[0][0] = 0;

   fseek(rom, 0x8be3e, SEEK_SET);

   for (int i = 1; i < 768; i++)
   {
      strLoc = 0;
      ch = 255;

      while (ch != 0)
      {
         ch = fgetc(rom);

         if (ch != 0)
            comprText[i][strLoc] = ch - 0x30;
         else
            comprText[i][strLoc] = 0;

         if (comprText[i][strLoc] == ' ')
             comprText[i][strLoc] = '_';

         strLoc++;
      }
   }
}

void LoadTextPointers(FILE* rom, TPointerEntry tpe[], int& tpeCount)
{
   // Assumes tpe has enough room, heh.  For now it's prolly safe to put it
   // in the 2000 range.  It'll be bigger later.
   // Fills out all the members of each tpe entry, including the strings.
   // This makes this easier to use, although it probably uses more memory
   // than it should, and might be pretty slow.

   // unsigned int  loc = 0x50200;
   unsigned int  loc = 0xf8b92;
   unsigned char ch;
   int           temp;

   fseek(rom, loc, SEEK_SET);

   tpeCount = 0;

   while (loc < 0xff4b0)
   {
      tpe[tpeCount].loc = loc;

      ch = fgetc(rom);
      ch = fgetc(rom);
      ch = fgetc(rom);
      ch = fgetc(rom);
      ch = fgetc(rom);

      tpe[tpeCount].spriteNum = fgetc(rom);
      tpe[tpeCount].spriteNum += (fgetc(rom) << 8);

      ch = fgetc(rom);
      tpe[tpeCount].movementNum = fgetc(rom);

      ch = fgetc(rom);
      ch = fgetc(rom);
      ch = fgetc(rom);
      ch = fgetc(rom);

      tpe[tpeCount].pointer = fgetc(rom);
      tpe[tpeCount].pointer += (fgetc(rom) << 8);
      tpe[tpeCount].pointer += (fgetc(rom) << 16);
      ch = fgetc(rom);
//      tpe[tpeCount].pointer += (fgetc(rom) << 24);

      temp = SNESToHex(tpe[tpeCount].pointer);

//      printf("%x\n", tpe[tpeCount].pointer);
      if ((temp >= 0x51d12) && (temp <= 0x4001ff))
      {
         GetStringAt(rom, temp, tpe[tpeCount].codeString, 1, tpe[tpeCount].size);
         GetStringAt(rom, temp, tpe[tpeCount].plainString, 0, tpe[tpeCount].size);

         tpeCount++;
      }

      loc += 17;
   }

/*
   while (loc < 0x51d12)
   {
      ch = fgetc(rom);
      ch = fgetc(rom);
      loc += 2;

      if (ch == 6)
      {
         ch = fgetc(rom);
         ch = fgetc(rom);
         ch = fgetc(rom);
         ch = fgetc(rom);
         ch = fgetc(rom);
         textPtrAddress[textPtrCount++] = loc - 2;

         loc += 5;
      }
   }*/
}

int GetType(unsigned char ch)
{
   // Returns the type of character ch represents in the text
   // stream.
   // Type 0:  A regular piece of text.
   // Type 1:  Means you'll need to use the lookup table
   // Type 2:  A control code

   char retVal = 2;

   if ((ch >= 0x50) && (ch <= 0xAA))
      retVal = 0;
   else if ((ch >= 0x15) && (ch <= 0x17))
      retVal = 1;

   return retVal;
}

void GetStringAt(FILE* rom, unsigned int address, char string[4096], int showCodes, int& size)
{
   // Gets the string located at the specified hex address.
   // Assumes rom is in read binary mode.

   unsigned char ch = 255;
   unsigned char ch2;
   char          tempStr[25];
   int           charType;
   int           strPos = 0;
   int           cNum;
   int           cLen;
   int           i;
   int           opCodeOn = 0;

   fseek(rom, address, SEEK_SET);

   for (i = 0; i < 14096; i++)
      string[i] = 0;

   size = 0;

   while ((ch != 2) && (size < 4000))
   {
      ch = fgetc(rom);
      charType = GetType(ch);

      if ((charType != 2) && (opCodeOn == 1) && showCodes)
      {
         string[strPos++] = ']';
         opCodeOn = 0;
      }

      switch (charType)
      {
         case 0:
         {
            string[strPos++] = ch - 48;
            size++;
            break;
         }
         case 1:
         {
            ch2 = fgetc(rom);

            cNum = (ch - 0x15) * 256 + ch2;
            cLen = strlen(comprText[cNum]);

            for (int i = 0; i < cLen; i++)
            {
               string[strPos++] = comprText[cNum][i];
            }

            size += 2;

            break;
         }
         case 2:
         {
            size++;

            if (opCodeOn == 0)
            {
               if (showCodes)
               {
                  sprintf(tempStr, "[%.2X", ch);
                  strcat(string, tempStr);
                  strPos += strlen(tempStr);
               }
               opCodeOn = 1;
            }
            else
            {
               if (showCodes)
               {
                  sprintf(tempStr, " %.2X", ch);
                  strcat(string, tempStr);
                  strPos += strlen(tempStr);
               }
            }

            if ((ch == 0x1C) || (ch == 0x08))
            {
               ch = fgetc(rom);
               ch2 = fgetc(rom);

               if (showCodes)
               {
                  sprintf(tempStr, " %.2X %.2X", ch, ch2);
                  strcat(string, tempStr);
                  strPos += strlen(tempStr);
               }
               size += 2;

               ch = 0;
            }
            else if ((ch == 0x19) || (ch == 0x04) || (ch == 0x05)
                  || (ch == 0x1F))
            {
               ch = fgetc(rom);

               if (showCodes)
               {
                  sprintf(tempStr, " %.2X", ch);
                  strcat(string, tempStr);
                  strPos += strlen(tempStr);
               }
               size++;

               ch = 0;
            }

            break;
         }
      }
   }

   if ((opCodeOn == 1) && showCodes)
   {
      string[strPos++] = ']';
   }
}

unsigned int HexToSNES(unsigned int address)
{
   // Converts good-ol regular hex addresses into the SNES-style addresses.
   // In this case, it's assumed the ROM is a hi-rom with a header.

   unsigned int retVal;

   retVal = (address + 0xc00000) - 0x200;

   return retVal;
}

unsigned int SNESToHex(unsigned int address)
{
   // Converts an SNES-style address (hi-rom, with header) to a nice
   // handy hex address.

   unsigned int retVal;

   retVal = (address - 0xc00000) + 0x200;

   return retVal;
}

void WriteText(FILE* rom, unsigned int address, char* string, int& size)
{
   // Writes the given string to the given address in the rom.
   // The address given is the hex address of the rom, not the SNES
   // address, so make sure to convert to hex before calling this.
   // size is the size of the string once it was written to the file.
   // Strings passed to this function are the very raw strings that you
   // get from users, such as:
   //
   // @Hi there![3 0]I am from Earth![3 0 2]

   char tempStr[1024];

   SimplifyString(string, tempStr, size);

   fseek(rom, address, SEEK_SET);
   fflush(rom);

   for (int i = 0; i < size; i++)
      fputc(tempStr[i], rom);
}

int ExpandROM(FILE* rom, unsigned int size)
{
   // Expands the rom to the given filesize.  Sort of assumes the
   // file has the file name given in the global string filename.
   // Returns 1 on success, 0 on failure.
   // 0x4001FF is the maximum size that is probably good; the text
   // addressing system can't access anything past of that.
   // That gives people about 1 extra meg to use for fun fun
   // text fun!

   long fileSize;
   int  retVal = 0;

   fseek(rom, 0, SEEK_END);
   fileSize = ftell(rom);

   if (fileSize < size)
   {
      fclose(rom);
//      fopen(filename, "ab");

      for (int i = fileSize; i < size; i++)
         fputc(0, rom);

      fclose(rom);

//      fopen(filename, "rb+");
      retVal = 1;
   }

   return retVal;
}

int GetNumbers(char* string, int number[256])
{
   // Assumes the string is not empty (0 size).
   // Assumes that string begins with '[' and ends with ']'.
   // Returns -1 if there's an error.
   // Assumes that all the numbers within are written in hex,
   // and are all valid hex numbers.

   int  strPos = 0;
   int  numPos = -1;
   int  strLen = strlen(string);
   int  total = 0;
   char ch;

   ch = string[strPos];
   if (ch != '[')
      return -1;

   while ((ch != ']') && (strPos < strLen))
   {
      ch = string[++strPos];

      if ((ch != ' ') && (ch != ']'))
      {
         number[++numPos] = 0;
         total++;
      }

      while ((ch != ' ') && (ch != ']'))
      {
         number[numPos] <<= 4;
         number[numPos] += CharToHex(ch);

         ch = string[++strPos];
      }
   }

   return total;
}

void SimplifyString(char* string, char newString[1024], int& size)
{
   // Simplifies a given string into what exactly will be written
   // to the rom.  Note that this doesn't write to the rom, it just
   // prepares the string.
   // newString isn't really a C-style string; since EarthBound doesn't
   // use 0 as the end-of-string identifier, you need to use the
   // size paramater to know where the string actually ends.

   char tempStr[1024];
   int  number[256];
   int  strPos = 0;
   int  tempPos;
   int  strLen;
   int  total;
   char ch;

   size = 0;

   strLen = strlen(string);

   while (strPos < strLen)
   {
      ch = string[strPos];

      if (ch == '[')
      {
         tempPos = 0;
         while (ch != ']')
         {
            tempStr[tempPos++] = ch;
            ch = string[++strPos];
         }

         tempStr[tempPos] = ']';
         tempStr[tempPos + 1] = '\0';

         total = GetNumbers(tempStr, number);

         for (int i = 0; i < total; i++)
            newString[size++] = number[i];
      }
      else
      {
         newString[size++] = ch + 0x30;
      }

      strPos++;
   }
}

int CharToHex(char ch)
{
   // Converts a single hex character to an integer.

   int retVal = 0;

   ch = toupper(ch);

   switch (ch)
   {
      case '0':
      {
         retVal = 0;
         break;
      }
      case '1':
      {
         retVal = 1;
         break;
      }
      case '2':
      {
         retVal = 2;
         break;
      }
      case '3':
      {
         retVal = 3;
         break;
      }
      case '4':
      {
         retVal = 4;
         break;
      }
      case '5':
      {
         retVal = 5;
         break;
      }
      case '6':
      {
         retVal = 6;
         break;
      }
      case '7':
      {
         retVal = 7;
         break;
      }
      case '8':
      {
         retVal = 8;
         break;
      }
      case '9':
      {
         retVal = 9;
         break;
      }
      case 'A':
      {
         retVal = 10;
         break;
      }
      case 'B':
      {
         retVal = 11;
         break;
      }
      case 'C':
      {
         retVal = 12;
         break;
      }
      case 'D':
      {
         retVal = 13;
         break;
      }
      case 'E':
      {
         retVal = 14;
         break;
      }
      case 'F':
      {
         retVal = 15;
         break;
      }
   }

   return retVal;
}
