#include <stdio.h>
#include <string.h>

struct Item
{
   void ChangeName(FILE*, char*);
   void ChangeType(FILE*, unsigned char);
   void ChangeCost(FILE*, int);
   void ChangeOwnership(FILE*, unsigned char);
   void ChangeEffect(FILE*, int);
   void ChangeStrength(FILE*, char);
   void ChangeIncrease(FILE*, unsigned char);
   void ChangeExtraPower(FILE*, char);
   void ChangeSpecialProperties(FILE*, unsigned char);
   void ChangeDescAddress(FILE*, unsigned int);

   long          address;
   char          name[25];
   unsigned char number;
   unsigned char type;
   int           cost;
   unsigned char ownership;
   int           effect;
   char          strength;
   unsigned char increase;
   char          extraPower;
   unsigned char specialProperties;
   unsigned int  descAddress;
};

void PullItems(FILE*, Item*);
void SortItemsAlpha(Item*);
void SortItemsType(Item*);
void SortItemsOriginal(Item*);
void SwapItem(Item&, Item&);
int SearchItem(Item*, char*, int[]);
void DumpItems(Item*, char*, int);


int main(int argc, char* argv[])
{
   FILE* rom;
   Item  item[256];
   int   searchList[256];
   int   numFound;
   int   i;
   int   j;

   if (argc < 2)
   {
      printf("Usage: pullitems rom.smc\n\n");
      return -1;
   }

   rom = fopen(argv[1], "rb+");
   if (!rom)
   {
      printf("Error opening %s, aborting.\n\n", argv[1]);
      return -1;
   }

   PullItems(rom, item);
   SortItemsAlpha(item);
   SortItemsType(item);
   DumpItems(item, "idump.txt", 1);

   fclose(rom);

   return 0;
}


void PullItems(FILE* rom, Item* item)
{
   // Assumes rom is in read binary
   // Assumes item is an array of 256 Item types

   char ch;
   int  offset = 0;

   fseek(rom, 0x155200, SEEK_SET);
   fflush(rom);

   for (int i = 0; i < 256; i++)
   {
      item[i].number = i;

      item[i].address = 0x155200 + offset;

      for (int j = 0; j < 25; j++)
      {
         ch = fgetc(rom);

         if (ch == 0)
            item[i].name[j] = 0;
         else
            item[i].name[j] = ch - 48;
      }

      item[i].type = fgetc(rom);
      item[i].cost = fgetc(rom);
      item[i].cost += (fgetc(rom) << 8);
      item[i].ownership = fgetc(rom);
      item[i].effect = fgetc(rom);
      item[i].effect += (fgetc(rom) << 8);
      item[i].strength = fgetc(rom);
      item[i].increase = fgetc(rom);
      item[i].extraPower = fgetc(rom);
      item[i].specialProperties = fgetc(rom);
      item[i].descAddress = fgetc(rom);
      item[i].descAddress += (fgetc(rom) << 8);
      item[i].descAddress += (fgetc(rom) << 16);
      item[i].descAddress += (fgetc(rom) << 24);

      offset += 39;
   }
}

void Item::ChangeName(FILE* rom, char* newName)
{
   // Assumes rom is in write binary
   // Assumes newName is at most 25 characters (last one being null)

   fseek(rom, address, SEEK_SET);
   fflush(rom);

   for (int i = 0; i < 25; i++)
   {
      if (newName[i] == 0)
         fputc(0, rom);
      else
         fputc(newName[i] + 48, rom);

      name[i] = newName[i];
   }
}

void Item::ChangeType(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Input values are limited, so please use the following chart:
   // 16 = Weapon
   // 20 = Body equipment
   // 24 = Arm Equipment
   // 28 = Other Equipment
   // 32 = if edible item, will say "... ate the ..."
   // 36 = if edible item, will say "... drank the ..."
   // 40 = condiment

   // Here are other possible #'s, but their effect on the game
   // don't appear to be noticeable, so don't implement these until
   // I figure them out better.
   // 0   = no idea (only the Franklin Badge uses it)
   // 2   = only one of the semi-non-existent items uses it, prolly useless
   // 4   = Teddy Bear flag
   // 8   = Broken item
   // 44  = large pizza?
   // 48  = healing item
   // 52  = weird item, seems to mostly be stuff that affects enemy status
   // 53  = no idea (dragonite, fly honey, etc.)
   // 56  = no idea (piggy nose, for sale sign, etc.)
   // 58  = no idea (bicycle, hawk eye, zombie paper only)
   // 59  = important event item
   // 255 = Null

   fseek(rom, address + 25, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   type = newValue;
}

void Item::ChangeCost(FILE* rom, int newValue)
{
   // Assumes rom is in write binary mode
   // newValue must be from 0 to 65535

   fseek(rom, address + 26, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   cost = newValue;
}

void Item::ChangeOwnership(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode.

   // This is somewhat more complex than other things, so pay
   // special attention.  Ownership refers to who can use what
   // item, and it also goes so far as to specify who can hold
   // what items.  I'll give some examples later.

   // newValue is 8 bits long.
   // bit:   _ _ _ _  _ _ _ _
   // bit #: 8 7 6 5  4 3 2 1

   // Bit 1 determines if Ness can use this item
   // Bit 2 determines if Paula can use this item
   // Bit 3 determines if Jeff can use this item
   // Bit 4 determines if Poo can use this item

   // Bit 5 is weird, I don't know what it does, but it's chicken related
   // Bit 6 and 7 are weird.  If both are 1, then once you get an item,
   // you can't give them to anybody else in your party (and obviously to
   // any non-playable characters).
   // Bit 8 signifies if the item disappears after use or not.

   // NOTE: If the item is food, anybody can eat it, regardless of
   //       ownership flags.

   // Here are some examples:

   // dec: 01  bin: 0000 0001 Only Ness can use this, and it doesn't
   //                         disappear after use.

   // dec: 111 bin: 0110 1111 once somebody gets this item, they're not
   //                         able to get rid of it!  Ahh!  BTW, this is
   //                         how the ATM Card, Sound Stone, and Receiver
   //                         Phone work.

   // dec: 104 bin: 0110 1000 only Poo can hold this item, and he can't
   //                         give it to any other party members.  This
   //                         is how the Tiny Ruby works.

   // dec: 132 bin: 1000 0100 This item is for Jeff, and disappears after
   //                         being used.  This is how Bottle Rockets work.

   // dec: 4   bin: 0000 0100 Only Jeff can use this item.  And it doesn't
   //                         disappear after you use it.  Wouldn't it
   //                         be cool if a Multi Bottle Rocket used this? :)

   // dec: 8   bin: 0000 1111 This can be used by everybody, and doesn't
   //                         disappear after use.  This can let everybody
   //                         have a Gutsy bat, or even have food that
   //                         doesn't disappear after you eat it!  Yahooooo!

   // Of course there are a lot more possibilities.

   fseek(rom, address + 28, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   ownership = newValue;
}

void Item::ChangeEffect(FILE* rom, int newValue)
{
   // Assumes rom is in write binary mode

   // Effect means what will happen when you use this item.
   // Like, you can make a Teddy Bear edible (although the game will
   // do some funky stuff when you try to eat it), or even make
   // something like a Cookie heal all members, like a Large Pizza
   // does.

   // What's cool about this is that you can make it so when you
   // use any item, say a Cookie, you can do stuff you normally
   // could only do in battle.  You can even make items do stuff
   // only enemies could do!  But sometimes using an item like
   // that will make you die instantly or the screen will garble up.
   // But it's cool nonetheless.

   // Valid codes are given in a file I've created called effects.txt.

   fseek(rom, address + 29, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   effect = newValue;
}

void Item::ChangeStrength(FILE* rom, char newValue)
{
   // Assumes rom is in write binary mode
   // Note: Values are from -127 to 127

   fseek(rom, address + 31, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   strength = newValue;
}

void Item::ChangeExtraPower(FILE* rom, char newValue)
{
   // This function changes what this item will "increase"
   // when used.  Like how cookies restore HP, while PSI
   // caramels restore PP, and Luck Capsules increase Luck.

   // Assumes rom is in write binary mode
   // Input values are limited, so please use the following chart:
   // 0  = HP
   // 1  = PP
   // 2  = HP & PP
   // 3  = Like rock candy (random stat + 1)
   // 4  = IQ
   // 5  = Guts
   // 6  = Speed
   // 7  = Vitality
   // 8  = Luck
   // 9  = No visible effect (that's what it says if you use it)

   // Many items use numbers other than these, but I can't tell
   // what they signify yet, so for now just use this chart.

   fseek(rom, address + 33, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   extraPower = newValue;
}

void Item::ChangeIncrease(FILE* rom, unsigned char newValue)
{
   // "extraPower" is related to "increase".
   // While "extraPower" describes what stats will be increased,
   // "increase" describes how much that stat will be increased.
   // Like when you equip the Gutsy Bat, your Guts go up 127.
   // Or some things like the Rabbit's Foot increase your luck.
   // Essentially this function tells how much to increase by.

   // Assumes rom is in write binary mode
   // Input values are from -127 to 127.

   fseek(rom, address + 32, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   increase = newValue;
}

void Item::ChangeSpecialProperties(FILE* rom, unsigned char newValue)
{
   // This function changes what special properties an item has.
   // Like how some items help protect you against Fire Attacks,
   // others against Hypnosis, etc.

   // Assumes rom is in write binary mode
   // I still haven't completely figured out the numbering system
   // for this.  But numbers vary from 0 - 255.

   fseek(rom, address + 34, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   specialProperties = newValue;
}

void Item::ChangeDescAddress(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are really unknown, oh well, that's
   // the user's problem.  Just let it be from 0 to FFFFFFFF.

   fseek(rom, address + 35, SEEK_SET);
   fflush(rom);

   fputc(newValue & 0xff, rom);
   fputc((newValue >> 8) & 0xff, rom);
   fputc((newValue >> 16) & 0xff, rom);
   fputc((newValue >> 24) & 0xff, rom);

   descAddress = newValue;
}

void SortItemsAlpha(Item* item)
{
   // Assumes item is allocated with at least 256 entries.
   // Only entries 0 through 255 will be rearranged alphabetically
   // (ascending order).
   // In the case that two items have the same name, they're rearranged
   // according to the "number" field, in ascending order.

   char currName[25];
   char bigName[25];
   int  biggest;
   int  result;
   int  i;
   int  j;

   for (i = 0; i < 255; i++)
   {
      biggest = i;

      for (j = i; j < 256; j++)
      {
         strcpy(bigName, item[biggest].name);
         strcpy(currName, item[j].name);

         strupr(bigName);
         strupr(currName);

         result = strcmp(currName, bigName);
         if (result < 0)
            biggest = j;
         else if (result == 0)
         {
            if (item[biggest].number > item[j].number)
               biggest = j;
         }
      }

      if (biggest != i)
         SwapItem(item[i], item[biggest]);
   }
}

void SortItemsType(Item* item)
{
   // Assumes item is allocated with at least 256 entries.
   // Only entries 0 through 255 will be rearranged in ascending order
   // by their "type" field.
   // In the case that two items have the same number, they're rearranged
   // by name.  If the names and types are the same, no special action
   // is taken, and they may be arranged in any order.

   char currName[25];
   char bigName[25];
   int  biggest;
   int  result;
   int  i;
   int  j;

   for (i = 0; i < 255; i++)
   {
      biggest = i;

      for (j = i; j < 256; j++)
      {
         if (item[biggest].type > item[j].type)
         {
            biggest = j;
         }
         else if (item[biggest].type == item[j].type)
         {
            strcpy(bigName, item[biggest].name);
            strcpy(currName, item[j].name);

            strupr(bigName);
            strupr(currName);

            result = strcmp(currName, bigName);
            if (result < 0)
               biggest = j;
         }
      }

      if (biggest != i)
         SwapItem(item[i], item[biggest]);
   }
}

void SortItemsOriginal(Item* item)
{
   // Assumes item is allocated with at least 256 entries.
   // Only entries 0 through 255 will be rearranged to their
   // in-ROM order.
   // In the case that two items have the same name, they're rearranged
   // according to the "number" field, in ascending order.

   int   biggest;
   int   i;
   int   j;

   for (i = 0; i < 255; i++)
   {
      biggest = i;

      for (j = i; j < 256; j++)
      {
         if (item[biggest].number > item[j].number)
            biggest = j;
      }

      if (biggest != i)
         SwapItem(item[i], item[biggest]);
   }
}


void SwapItem(Item& item1, Item& item2)
{
   Item temp;

   temp = item1;
   item1 = item2;
   item2 = temp;
}



int SearchItem(Item* item, char* searchStr, int results[])
{
   int index = 0;

   // Assumes items has been allocated with 256 entries.
   // This function will locate all items whose names contain
   // searchStr.  Each successful matched enemy will have its number
   // placed into the results array.
   // This function returns the number of matches that were found.
   // Only entries from 0...returnValue - 1 (inclusive) are guaranteed
   // to be correct, all other entries in the results array will be garbage.

   for (int i = 0; i < 256; i++)
   {
       if (strstr(item[i].name, searchStr) != NULL)
          results[index++] = i;
   }

   return index;
}

void DumpItems(Item* item, char* filename, int stats)
{
   // Dumps all the item information to a text file.
   // If stats is non-zero, all the stats of each item gets
   // printed out.  If it's zero, just the names and numbers of each
   // item get printed out.

   FILE* dumpFile;

   dumpFile = fopen(filename, "w");

   fprintf(dumpFile, "EarthBound item info dump generated by PK Hack\n");
   fprintf(dumpFile, "PK Hack by Tomato (tomato@starmen.net)");
   fprintf(dumpFile, " and spaanoft (spaanoft@starmen.net)\n");
   fprintf(dumpFile, "--------------------------------------");
   fprintf(dumpFile, "------------------------------------\n\n");

   for (int i = 0; i < 256; i++)
   {
      fprintf(dumpFile, "Item #:        %d\n", item[i].number);
      fprintf(dumpFile, "Name:          %s\n", item[i].name);

      if (stats)
      {
         fprintf(dumpFile, "Effect:        %d\n", item[i].effect);
         fprintf(dumpFile, "Cost           $%d\n", item[i].cost);
         fprintf(dumpFile, "Type:          %d\n", item[i].type);
         fprintf(dumpFile, "Strength:      %d\n", item[i].strength);
         fprintf(dumpFile, "Extra Power:   %d\n", item[i].extraPower);
         fprintf(dumpFile, "Increase #:    %d\n", item[i].increase);
         fprintf(dumpFile, "Special Prop.: %d\n", item[i].specialProperties);
         fprintf(dumpFile, "Desc. Address: 0x%x\n", item[i].descAddress);
         fprintf(dumpFile, "Ownership:     0x%.2x\n", item[i].ownership);
       }

       fprintf(dumpFile, "\n");
   }

   fclose(dumpFile);
}
