#include <stdio.h>
#include <string.h>
#include "effects.h"
#include "bgm.h"
#include "eorder.h"

#define TYPE_NORMAL 0
#define TYPE_PSI    1
#define TYPE_CALL   2
#define TYPE_ITEM   3

struct Enemy
{
   void ChangeTheFlag(FILE*, unsigned char);
   void ChangeName(FILE*, char*);
   void ChangeInsidePic(FILE*, unsigned char);
   void ChangeOutsidePic(FILE*, int);
   void ChangeRunFlag(FILE*, unsigned char);
   void ChangeHp(FILE*, unsigned int);
   void ChangePp(FILE*, unsigned int);
   void ChangeExp(FILE*, unsigned int);
   void ChangeMoney(FILE*, unsigned int);
   void ChangeSpeed(FILE*, unsigned char);
   void ChangeStartAction(FILE*, unsigned int);
   void ChangeDieAction(FILE*, unsigned int);
   void ChangePalette(FILE*, unsigned char);
   void ChangeOffense(FILE*, unsigned char);
   void ChangeDefense(FILE*, unsigned char);
   void ChangeItem(FILE*, unsigned char);
   void ChangeFreq(FILE*, unsigned char);
   void ChangeStatus(FILE*, unsigned char);
   void ChangeAction(FILE*, int, int);
   void ChangeFinalAction(FILE*, int);
   void ChangeLevel(FILE*, unsigned char);
   void ChangeGuts(FILE*, unsigned char);
   void ChangeIq(FILE*, unsigned char);
   void ChangeMovement(FILE*, unsigned char);
   void ChangeOrder(FILE*, unsigned char);
   void ChangeArgument(FILE*, unsigned char, unsigned char);
   void ChangeMaxCall(FILE*, unsigned char);
   void ChangeMissRate(FILE*, unsigned char);
   void ChangeDieSound(FILE*, unsigned char);
   void ChangeType(FILE*, unsigned char);
   void ChangeMusic(FILE*, unsigned char);
   void ChangeBossFlag(FILE*, unsigned char);
   int  GetActionType(int);

   long          address;
   unsigned char theFlag;
   char          name[25];
   unsigned char number;
   unsigned char insidePic;
   int           outsidePic;
   unsigned char runFlag;
   unsigned int  hp;
   unsigned int  pp;
   unsigned int  exp;
   unsigned int  money;
   unsigned char speed;
   unsigned int  startAddress;
   unsigned int  dieAddress;
   unsigned char palette;
   unsigned char offense;
   unsigned char defense;
   unsigned char item;
   unsigned char freq;
   unsigned char status;
   int           action[4];
   int           finalAction;
   unsigned char level;
   unsigned char guts;
   unsigned char iq;
   unsigned char bossFlag;  // still not worked out fully
   unsigned char dieSound;
   unsigned char movement;
   unsigned char order;
   unsigned char argument[4];
   unsigned char maxCall;
   unsigned char missRate;
   unsigned char type;
   unsigned char music;
   unsigned char unknown;
};
  
void PullEnemies(FILE*, Enemy*);
void SortEnemiesOriginal(Enemy*);
void SortEnemiesAlpha(Enemy*);
void SortEnemiesGameOrder(Enemy*);
int  SearchEnemy(Enemy*, char*, int[]);
void SwapEnemy(Enemy&, Enemy&);
void DumpEnemies(Enemy*, char*, int, int, int);
void PullPSIList(FILE*, char[51][30]);

int main(int argc, char* argv[])
{
   struct Enemy enemy[230];
   FILE*  rom;

   if (argc < 2)
   {
      printf("Usage: enemy rom.smc\n\n");
      return -1;
   }

   rom = fopen(argv[1], "rb+");
   if (!rom)
   {
      printf("Error opening %s, aborting.\n\n", argv[1]);
      return -1;
   }

   PullEnemies(rom, enemy);
   SortEnemiesGameOrder(enemy);

   DumpEnemies(enemy, "edump.txt", 1, 1, 1);

   fclose(rom);

   return 0;
}


void PullEnemies(FILE* rom, struct Enemy* enemy)
{
   long int      offset = 0;
   int           a;
   int           b;
   unsigned char ch;

   fseek(rom, 0x1597e7, SEEK_SET);
   fflush(rom);

   for (a = 0; a < 230; a++)
   {
      enemy[a].number = a;
      enemy[a].address = 1415143 + offset;
      enemy[a].theFlag = fgetc(rom);

      for (b = 0; b < 25; b++)
      {
         ch = fgetc(rom);

         if (ch == 0)
            enemy[a].name[b] = 0;
         else
            enemy[a].name[b] = ch - 48;
      }

      ch = fgetc(rom);  // Unknown, varies from 0-3 or so
                        // unknown5.txt

      enemy[a].type = fgetc(rom); // 0 - normal, 1 - insect, 2 - metal

      enemy[a].insidePic = fgetc(rom);
      enemy[a].insidePic += (fgetc(rom) << 8);

      enemy[a].outsidePic = fgetc(rom);
      enemy[a].outsidePic += (fgetc(rom) << 8);

      enemy[a].runFlag = fgetc(rom);

      enemy[a].hp = fgetc(rom);
      enemy[a].hp += (fgetc(rom) * 256);

      enemy[a].pp = fgetc(rom);
      enemy[a].pp += (fgetc(rom) * 256);

      enemy[a].exp = fgetc(rom);
      enemy[a].exp += (fgetc(rom) * 256);
      enemy[a].exp += (fgetc(rom) * 256 * 256);
      enemy[a].exp += (fgetc(rom) * 256 * 256 * 256);

      enemy[a].money = fgetc(rom);
      enemy[a].money += (fgetc(rom) * 256);

      enemy[a].movement = fgetc(rom);
      ch = fgetc(rom); // Appears to always be 0

      enemy[a].startAddress = fgetc(rom);
      enemy[a].startAddress += (fgetc(rom) << 8);
      enemy[a].startAddress += (fgetc(rom) << 16);
      enemy[a].startAddress += (fgetc(rom) << 24);

      enemy[a].dieAddress = fgetc(rom);
      enemy[a].dieAddress += (fgetc(rom) << 8);
      enemy[a].dieAddress += (fgetc(rom) << 16);
      enemy[a].dieAddress += (fgetc(rom) << 24);

      enemy[a].palette = fgetc(rom);

      enemy[a].level = fgetc(rom);

      enemy[a].music = fgetc(rom);

      enemy[a].offense = fgetc(rom);

      ch = fgetc(rom); // Appears to always be 0

      enemy[a].defense = fgetc(rom);

      ch = fgetc(rom);  // Unknown, but mostly 0, sometimes 1
                        // unknown4.txt

      enemy[a].speed = fgetc(rom);
      enemy[a].guts = fgetc(rom);
      enemy[a].iq = fgetc(rom);

      ch = fgetc(rom);  // Appears to vary from 0-4 or so
      ch = fgetc(rom);  // Appears to vary from 0-3
      ch = fgetc(rom);  // Appears to vary from 0-3
      ch = fgetc(rom);  // Appears to vary from 0-3
      ch = fgetc(rom);  // Appears to vary from 0-3
      enemy[a].missRate = fgetc(rom);  // 0 = almost never, 255 = always

      enemy[a].order = fgetc(rom);

      enemy[a].action[0] = fgetc(rom);
      enemy[a].action[0] += (fgetc(rom) << 8);
      enemy[a].action[1] = fgetc(rom);
      enemy[a].action[1] += (fgetc(rom) << 8);
      enemy[a].action[2] = fgetc(rom);
      enemy[a].action[2] += (fgetc(rom) << 8);
      enemy[a].action[3] = fgetc(rom);
      enemy[a].action[3] += (fgetc(rom) << 8);

      enemy[a].finalAction = fgetc(rom);
      enemy[a].finalAction += (fgetc(rom) << 8);

      enemy[a].argument[0] = fgetc(rom);
      enemy[a].argument[1] = fgetc(rom);
      enemy[a].argument[2] = fgetc(rom);
      enemy[a].argument[3] = fgetc(rom);

      ch = fgetc(rom);  // dunno, always 0, except Carbon Dog, who has 83

      ch = fgetc(rom);  // seems to be size, or maybe precedence, possibly
                        // linked to calling enemies, unknown3.txt

      enemy[a].bossFlag = fgetc(rom);

      enemy[a].freq = fgetc(rom);
      enemy[a].item = fgetc(rom);

      enemy[a].status = fgetc(rom);
      enemy[a].dieSound = fgetc(rom);

      ch = fgetc(rom);  // Has to do with battle layout, 0 or 1
                        // unknown2.txt

      enemy[a].maxCall = fgetc(rom);  // Very max = 8
      
      ch = fgetc(rom);  // Unknown, numbers vary somewhat, maybe enemy type
                        // unknown1.txt

      offset += 94;
   }
}


void Enemy::ChangeTheFlag(FILE* rom, unsigned char newValue)
{
   // rom must already be open, and in write mode
   // is assumed newValue is 0 or 1

   // if newValue is 1, then this enemy will have the word "the"
   // displayed in front of its name during battle

   fseek(rom, address, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   theFlag = newValue;
}


void Enemy::ChangeName(FILE* rom, char* newName)
{
   // Assumes rom is in write binary mode
   // Assumes newName is 25 characters long, last one is 0

   fseek(rom, address + 1, SEEK_SET);
   fflush(rom);

   for (int i = 0; i < 25; i++)
   {
      if (newName[i] == 0)
      {
         fputc(0, rom);
         name[i] = 0;
      }
      else
      {
         fputc(newName[i] + 48, rom);
         name[i] = newName[i] + 48;
      }
   }
}


void Enemy::ChangeInsidePic(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are found in a file called inside.txt.
   // It may turn out that the inside pic is decided using an address
   // instead of an array index.  I'll keep you posted.

   fseek(rom, address + 28, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   insidePic = newValue;
}


void Enemy::ChangeOutsidePic(FILE* rom, int newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are written in a text file called
   // outside.txt

   fseek(rom, address + 30, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   outsidePic = newValue;
}


void Enemy::ChangeRunFlag(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Have no idea of possible #'s, except that only 6 and 7 are used
   // in EarthBound.  I'll figure this out later.

   fseek(rom, address + 32, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   runFlag = newValue;
}


void Enemy::ChangeHp(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Assumes integer is from 0 to 65535

   fseek(rom, address + 33, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   hp = newValue;
}


void Enemy::ChangePp(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Assumes integer is from 0 to 65535

   fseek(rom, address + 35, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   pp = newValue;
}


void Enemy::ChangeExp(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Assumes integer is from 0 to 2^32 (over 4 billion!)

   fseek(rom, address + 37, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255 , rom);
   fputc((newValue >> 16) & 255, rom);
   fputc((newValue >> 24) & 255, rom);

   exp = newValue;
}


void Enemy::ChangeMoney(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Assumes integer is from 0 to 65535

   fseek(rom, address + 41, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   money = newValue;
}


void Enemy::ChangeSpeed(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are from 0 - 255

   fseek(rom, address + 60, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   speed = newValue;
}


void Enemy::ChangePalette(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Valid #'s are from 0 - 31
   // Maybe later I'll have exact palette info for ya

   fseek(rom, address + 53, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   palette = newValue;
}


void Enemy::ChangeOffense(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are from 0-255

   fseek(rom, address + 56, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   offense = newValue;
}


void Enemy::ChangeDefense(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are from 0-255

   fseek(rom, address + 58, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   defense = newValue;
}


void Enemy::ChangeItem(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are 0-255, but 254 and 255 are sorta non-existent
   // items (at least in EB...)

   fseek(rom, address + 88, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   item = newValue;
}


void Enemy::ChangeFreq(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are from 0-7

   // The frequency equation seems to work this way:
   // drop percentage = (2^newValue)/128 of a chance
   // Thanks for bringing that to my attention, Spiffage ;)

   fseek(rom, address + 87, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   freq = newValue;
}


void Enemy::ChangeStatus(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are found in a file called enstatus.txt.

   fseek(rom, address + 89, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   status = newValue;
}


void Enemy::ChangeAction(FILE* rom, int newValue, int index)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are shown in effects.txt
   // Allowed values for index are 0-3.  index is the specific action
   // to be edited.  i.e. action[0], action[1], action[2], or action[3]

   fseek(rom, address + 70 + (index * 2), SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   action[index] = newValue;
}


void Enemy::ChangeFinalAction(FILE* rom, int newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are listed in effects.txt

   fseek(rom, address + 78, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);

   finalAction = newValue;
}


void Enemy::ChangeLevel(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are 0 to 255

   fseek(rom, address + 54, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   level = newValue;
}


void Enemy::ChangeGuts(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are 0 to 255

   fseek(rom, address + 61, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   guts = newValue;
}


void Enemy::ChangeIq(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are 0 to 255

   fseek(rom, address + 62, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   iq = newValue;
}


void Enemy::ChangeStartAction(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue appear to be 0 to 2^32.  But not all
   // values will be happy.  newValue acts sort of as a pointer to
   // the game text.  Except I haven't figured out why it's a 32 bit
   // address or number.

   fseek(rom, address + 45, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);
   fputc((newValue >> 16) & 255, rom);
   fputc((newValue >> 24) & 255, rom);

   startAddress = newValue;
}


void Enemy::ChangeDieAction(FILE* rom, unsigned int newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue appear to be 0 to 2^32.  Works like
   // ChangeStartAction.

   fseek(rom, address + 49, SEEK_SET);
   fflush(rom);

   fputc(newValue & 255, rom);
   fputc((newValue >> 8) & 255, rom);
   fputc((newValue >> 16) & 255, rom);
   fputc((newValue >> 24) & 255, rom);

   dieAddress = newValue;
}

void Enemy::ChangeMovement(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values for newValue are given in movement.txt
   // Note that any values other than listed in that text file
   // may result in unplayable roms and or totally screwy stuff
   // may occur.

   fseek(rom, address + 43, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   movement = newValue;
}

void Enemy::ChangeOrder(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Changes the order in which enemies attack, depending on
   // newValue, which can be one of 4 numbers:
   // 0 - Random
   // 1 - Random, but tends to do the 3rd attack often
   // 2 - In order (Attack 1, Attack 2, Attack 3, Attack 4, repeat...)
   // 3 - "Staggered Order" (difficult to explain, just it goes something
   //     like "Attack 3, 4, 3, 4, 3, 4, 3, 1, 2, 1, 2, etc.  It's not
   //     a very logical pattern, but it's a pattern.  This is what Buzz
   //     Buzz does when he's in your party, and what other enemies do to.
   //     Just call it "Staggered Order" in the GUI, cuz it's too hard
   //     to figure out a better name for it.

   fseek(rom, address + 69, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   order = newValue;
}

void Enemy::ChangeArgument(FILE* rom, unsigned char slot, unsigned char enVal)
{
   // Assumes rom is in write binary mode
   // slot is from 0 to 3, which action's argument to edit
   // possible input values vary depending on the type of action associated
   // with the given slot

   fseek(rom, address + 80 + slot, SEEK_SET);
   fflush(rom);

   fputc(enVal, rom);
 
   argument[slot] = enVal;
}

void Enemy::ChangeMaxCall(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // newValue can be from 0 to 255, but the game will only allow a max of
   // 8 it seems. Anything over 8 seems to be treated as 8.  Best to leave
   // this fully editable with a text field instead of a dropdown list.
   // This function changes how many called enemies there can be total.

   fseek(rom, address + 92, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   maxCall = newValue;
}

void Enemy::ChangeMissRate(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // newValue can be from 0 to 255.  0 means never miss, 255 means always
   // miss.  This is when they attack you.  Anyway, I'm not completely sure
   // how this figures in with your speed setting or whatever.  Perhaps there
   // are supposed to be enemies that always hit you, no matter what your
   // speed or something.

   fseek(rom, address + 68, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   missRate = newValue;
}

void Enemy::ChangeDieSound(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // newValue can be 0 or 1.  Other values *may* be allowed, but all the
   // other ones I tried just did the same as if it were just 1.
   // This changes the sound as the enemy is fading away.  0 makes it sound
   // like how a normal enemy dies.  1 makes it sound a bit like how bosses
   // die.

   fseek(rom, address + 90, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   dieSound = newValue;
}

void Enemy::ChangeType(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are from 0 - normal, 1 - insect, 2 - metal

   fseek(rom, address + 27, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   type = newValue;
}

void Enemy::ChangeMusic(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary form
   // Allowed values are from 0 to 191

   fseek(rom, address + 55, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   music = newValue;
}

void Enemy::ChangeBossFlag(FILE* rom, unsigned char newValue)
{
   // Assumes rom is in write binary mode
   // Allowed values are 0 (not a boss) to 1 (a boss)
   // However, changing this flag so far appears to have no effect
   // on the game.

   fseek(rom, address + 86, SEEK_SET);
   fflush(rom);

   fputc(newValue, rom);

   bossFlag = newValue;
}


void SortEnemiesAlpha(Enemy* enemy)
{
   // Assumes enemy is allocated with at least 230 entries.
   // Only entries 0 through 229 will be rearranged alphabetically
   // (ascending order).
   // In the case that two enemies have the same name, they're rearranged
   // according to the "number" field, in ascending order.
   // No complaining anymore, spaan! ;)

   Enemy temp;
   char  currName[25];
   char  bigName[25];
   int   biggest;
   int   result;
   int   tempNum;
   int   i;
   int   j;

   for (i = 0; i < 229; i++)
   {
      biggest = i;

      for (j = i; j < 230; j++)
      {
         strcpy(bigName, enemy[biggest].name);
         strcpy(currName, enemy[j].name);

         strupr(bigName);
         strupr(currName);

         result = strcmp(currName, bigName);
         if (result < 0)
            biggest = j;
         else if (result == 0)
         {
            if (enemy[biggest].number > enemy[j].number)
               biggest = j;
         }
      }

      if (biggest != i)
         SwapEnemy(enemy[i], enemy[biggest]);
   }
}

void SortEnemiesOriginal(Enemy* enemy)
{
   // Assumes enemy is allocated with at least 230 entries.
   // Only entries 0 through 229 will be rearranged.
   // The enemy array gets rearranged to the original order as the items
   // appear in the ROM.  In other words, the order you get is exactly
   // the same as if you had just called PullEnemies.

   Enemy temp;
   char  currName[25];
   char  bigName[25];
   int   biggest;
   int   result;
   int   tempNum;
   int   i;
   int   j;

   for (i = 0; i < 229; i++)
   {
      biggest = i;

      for (j = i; j < 230; j++)
      {
         if (enemy[biggest].number > enemy[j].number)
            biggest = j;
      }

      if (biggest != i)
         SwapEnemy(enemy[i], enemy[biggest]);
   }
}

void SortEnemiesGameOrder(Enemy* enemy)
{
   // Sorts this enemy array into the order listed in eorder.h
   // Assumes there are at least 230 entries allocated in the array
   // Only entries 0 thru 229 are sorted.

   int i;
   int j;

   for (i = 0; i < 229; i++)
   {
      for (j = i; j < 230; j++)
      {
         if (enemy[j].number == enemyGameOrder[i])
            SwapEnemy(enemy[j], enemy[i]);
      }
   }
}


int SearchEnemy(Enemy* enemy, char* searchStr, int results[])
{
   int index = 0;

   // Assumes enemy has been allocated with 230 entries.
   // This function will locate all enemies whose names contain
   // searchStr.  Each successful matched enemy will have its number
   // placed into the results array.
   // This function returns the number of matches that were found.
   // Only entries from 0...returnValue - 1 (inclusive) are guaranteed
   // to be correct, all other entries in the results array will be garbage.

   for (int i = 0; i < 230; i++)
   {
       if (strstr(enemy[i].name, searchStr) != NULL)
          results[index++] = i;
   }

   return index;
}

void SwapEnemy(Enemy& enemy1, Enemy& enemy2)
{
   Enemy temp;

   temp = enemy1;
   enemy1 = enemy2;
   enemy2 = temp;
}

void DumpEnemies(Enemy* enemy, char* fName, int stats, int actions, int gfx)
{
   // Dumps enemy information into a text file.
   // If stats is non-zero, stats info will be printed.
   // If actions is non-zero, action info will be printed.
   // If gfx is non-zero, gfx info will be printed.

   FILE* dumpFile;

   dumpFile = fopen(fName, "w");

   fprintf(dumpFile, "EarthBound enemy info dump generated by PK Hack\n");
   fprintf(dumpFile, "PK Hack by Tomato (tomato@starmen.net)");
   fprintf(dumpFile, " and spaanoft (spaanoft@starmen.net)\n");
   fprintf(dumpFile, "--------------------------------------");
   fprintf(dumpFile, "------------------------------------\n\n");

   for (int i = 0; i < 230; i++)
   {
      fprintf(dumpFile, "Enemy #:   %d\n", enemy[i].number);
      fprintf(dumpFile, "Name:      %s\n", enemy[i].name);

      if (stats)
      {
         fprintf(dumpFile, "the Flag:  %d\n", enemy[i].theFlag);
         fprintf(dumpFile, "HP:        %d\n", enemy[i].hp);
         fprintf(dumpFile, "PP:        %d\n", enemy[i].pp);
         fprintf(dumpFile, "Exp:       %u\n", enemy[i].exp);
         fprintf(dumpFile, "Money:     $%d\n", enemy[i].money);
         fprintf(dumpFile, "Speed:     %d\n", enemy[i].speed);
         fprintf(dumpFile, "Offense:   %d\n", enemy[i].offense);
         fprintf(dumpFile, "Defense:   %d\n", enemy[i].defense);
         fprintf(dumpFile, "Status:    %d\n", enemy[1].status);
         fprintf(dumpFile, "Level:     %d\n", enemy[i].level);
         fprintf(dumpFile, "Guts:      %d\n", enemy[i].guts);
         fprintf(dumpFile, "IQ:        %d\n", enemy[i].iq);
         fprintf(dumpFile, "Miss rate: %d\n", enemy[i].missRate);
         fprintf(dumpFile, "Item:      %d\n", enemy[i].item);
         fprintf(dumpFile, "Drop %%:    %d\n", enemy[i].freq);
         fprintf(dumpFile, "Type:      %d\n", enemy[i].type);
         fprintf(dumpFile, "Music:     %s\n", bgm[enemy[i].music]);
         fprintf(dumpFile, "Die Sound: %d\n", enemy[i].dieSound);
         fprintf(dumpFile, "Is a Boss: %d\n", enemy[i].bossFlag);
      }

      if (actions)
      {
         fprintf(dumpFile, "Action 1:  %s\n", action[enemy[i].action[0]]);
         fprintf(dumpFile, "Action 2:  %s\n", action[enemy[i].action[1]]);
         fprintf(dumpFile, "Action 3:  %s\n", action[enemy[i].action[2]]);
         fprintf(dumpFile, "Action 4:  %s\n", action[enemy[i].action[3]]);
         fprintf(dumpFile, "Order:     %d\n", enemy[i].order);
         fprintf(dumpFile, "F. Action: %s\n", action[enemy[i].finalAction]);

         fprintf(dumpFile, "Arg. 1:    %d\n", enemy[i].argument[0]);
         fprintf(dumpFile, "Arg. 2:    %d\n", enemy[i].argument[1]);
         fprintf(dumpFile, "Arg. 3:    %d\n", enemy[i].argument[2]);
         fprintf(dumpFile, "Arg. 4:    %d\n", enemy[i].argument[3]);

         fprintf(dumpFile, "Max Call:  %d\n", enemy[i].maxCall);

         fprintf(dumpFile, "S Address: 0x%x\n", enemy[i].startAddress);
         fprintf(dumpFile, "D Address: 0x%x\n", enemy[i].dieAddress);
      }

      if (gfx)
      {
         fprintf(dumpFile, "Out Pic #: %d\n", enemy[i].outsidePic);
         fprintf(dumpFile, "Movement:  %d\n", enemy[i].movement);
         fprintf(dumpFile, "In Pic #:  %d\n", enemy[i].insidePic);
         fprintf(dumpFile, "Palette #: %d\n", enemy[i].palette);
      }

      fprintf(dumpFile, "\n");
   }

   fclose(dumpFile);
}

int GetActionType(int actionNum)
{
   // This returns a value which indicates what type of action this is.
   // This is helpful when determining what kind of arguments an action
   // should take.  It's assumed actionNum is within the appropriate
   // range of numbers, meaning 0 to the number of actions there happen to
   // be.
   // Return values are:
   // TYPE_NORMAL = This action is normal, no arguments needed
   // TYPE_PSI    = This action is PSI
   // TYPE_CALL   = This action is for calling other enemies
   // TYPE_ITEM   = This action is item-related

   #include "actiontype.h"

   return actionType[actionNum];
}
